/***************************************************************************/
/*                                                                         */
/*  truetype.c                                                             */
/*                                                                         */
/*    FreeType TrueType driver component (body only).                      */
/*                                                                         */
/*  Copyright 1996-2001, 2004, 2006 by                                     */
/*  David Turner, Robert Wilhelm, and Werner Lemberg.                      */
/*                                                                         */
/*  This file is part of the FreeType project, and may only be used,       */
/*  modified, and distributed under the terms of the FreeType project      */
/*  license, LICENSE.TXT.  By continuing to use, modify, or distribute     */
/*  this file you indicate that you have read the license and              */
/*  understand and accept it fully.                                        */
/*                                                                         */
/***************************************************************************/


#define FT_MAKE_OPTION_SINGLE_OBJECT

#include <ft2build.h>
#include "ttdriver.c"   /* driver interface    */
#include "ttpload.c"    /* tables loader       */
#include "ttgload.c"    /* glyph loader        */
#include "ttobjs.c"     /* object manager      */

#ifdef TT_USE_BYTECODE_INTERPRETER
#include "ttinterp.c"
#endif

#ifdef TT_CONFIG_OPTION_GX_VAR_SUPPORT
#include "ttgxvar.c"    /* gx distortable font */
#endif

#define		WALPHA_MAX				255


#include <scene.h>
#include <display.h>
#include <DrvMemMgr.h>
#include <fs_api.h>
#include <ttfont.h>

#include <debugoff.h>

p_ttf ettf = NULL, cttf = NULL;

/**
 * TTF
 * @param filename TTFļ
 * @param size ԤС
 * @return TTFָ
 * - NULL ʧ
 */
extern p_ttf ttf_open(const char *filename, int size)
{
	p_ttf ttf;
	byte *buf;
	int fileSize;

	if (filename == NULL || size == 0)
		return NULL;

	FILE *fp = fopen( filename, "rb" );
	if( fp == NULL )
		return NULL;

	fseek( fp, 0, SEEK_END );

	fileSize = ftell( fp );

	fseek( fp, 0, SEEK_SET );

	buf = (byte *) malloc(fileSize);
	if (buf == NULL) {
		return NULL;
	}
	fread( buf, 1, fileSize, fp );
	fclose( fp );
	ttf = ttf_open_buffer(buf, fileSize, size, filename);
	return ttf;
}

/**
 * TTF
 * @param ttfBuf TTF
 * @param ttfLength TTFݴСֽڼ
 * @param pixelSize ԤС
 * @param ttfName TTF
 * @return TTFָ
 * - NULL ʧ
 */
extern p_ttf ttf_open_buffer(void *ttfBuf, size_t ttfLength, int pixelSize,
							 const char *ttfName)
{
	int i;
	p_ttf ttf;

	if (ttfBuf == NULL || ttfLength == 0 || ttfName == NULL){
		return NULL;
	}
	if ((ttf = (p_ttf) calloc(1, sizeof(t_ttf))) == NULL) {
		return NULL;
	}

	memset(ttf, 0, sizeof(t_ttf));
	ttf->fileBuffer = ttfBuf;
	ttf->fileSize = ttfLength;
	if (strrchr(ttfName, '/') == NULL)
		ttf->fontName = strdup(ttfName);
	else {
		ttf->fontName = strdup(strrchr(ttfName, '/') + 1);
	}

	if (FT_Init_FreeType(&ttf->library) != 0) {
		free(ttf->fileBuffer);
		free(ttf);
		return NULL;
	}
	if (FT_New_Memory_Face
		(ttf->library, ttf->fileBuffer, ttf->fileSize, 0, &ttf->face)
		!= 0) {
		FT_Done_FreeType(ttf->library);
		free(ttf->fileBuffer);
		free(ttf);
		return NULL;
	}
	for (i = 0; i < SBIT_HASH_SIZE; ++i) {
		memset(&ttf->sbitHashRoot[i], 0, sizeof(SBit_HashItem));
	}
	ttf->cacheSize = 0;
	ttf->cachePop = 0;
	ttf_set_pixel_size(ttf, pixelSize);
	return ttf;
}

/**
 * رTTFṹ
 * @param ttf ttfָ
 */
extern void ttf_close(p_ttf ttf)
{
	int i;

	if (ttf == NULL)
		return;

	if (ttf->fontName != NULL) {
		free(ttf->fontName);
		ttf->fontName = NULL;
	}
	if (ttf->fileBuffer != NULL) {
		free(ttf->fileBuffer);
		ttf->fileBuffer = NULL;
	}

	for (i = 0; i < SBIT_HASH_SIZE; ++i) {
		if (ttf->sbitHashRoot[i].bitmap.buffer) {
			free(ttf->sbitHashRoot[i].bitmap.buffer);
			ttf->sbitHashRoot[i].bitmap.buffer = NULL;
		}
	}

	FT_Done_Face(ttf->face);
	FT_Done_FreeType(ttf->library);

	free(ttf);
}

/**
 * TTFС
 * @param ttf ttfָ
 * @param size С
 * @return Ƿɹ
 */
extern BOOL ttf_set_pixel_size(p_ttf ttf, int size)
{
	if (ttf == NULL)
		return false;

	if (FT_Set_Pixel_Sizes(ttf->face, 0, size) != 0) {
		return false;
	}

	ttf->pixelSize = size;
	return true;
}

/**
 * TTFǷÿЧ
 * @param ttf ttfָ
 * @param aa ǷÿЧ
 */
extern void ttf_set_anti_alias(p_ttf ttf, BOOL aa)
{
	if (ttf == NULL)
		return;

	ttf->antiAlias = aa;
}

/**
 * TTFǷClearType(Sub-Pixel LCDŻЧ)
 * @param ttf ttfָ
 * @param cleartype Ƿcleartype
 */
extern void ttf_set_cleartype(p_ttf ttf, BOOL cleartype)
{
	if (ttf == NULL)
		return;

	if (cleartype)
		FT_Library_SetLcdFilter(ttf->library, FT_LCD_FILTER_DEFAULT);
	ttf->cleartype = cleartype;
}

/**
 * TTFǷӴ
 * @param ttf ttfָ
 * @param embolden ǷӴ
 */
extern void ttf_set_embolden(p_ttf ttf, BOOL embolden)
{
	if (ttf == NULL)
		return;

	ttf->embolden = embolden;
}

/**
 * εttfͻ
 */
static void sbitCacheAdd(p_ttf ttf, unsigned long ucsCode, int glyphIndex,
						 FT_Bitmap * bitmap, int left, int top, int xadvance,
						 int yadvance)
{
	int addIndex = 0;

	if (ttf->cacheSize < SBIT_HASH_SIZE) {
		addIndex = ttf->cacheSize++;
	} else {
		addIndex = ttf->cachePop++;
		if (ttf->cachePop == SBIT_HASH_SIZE)
			ttf->cachePop = 0;
	}
	SBit_HashItem *item = &ttf->sbitHashRoot[addIndex];

	if (item->bitmap.buffer) {
		free(item->bitmap.buffer);
		item->bitmap.buffer = 0;
	}

	item->ucs_code = ucsCode;
	item->glyph_index = glyphIndex;
	item->size = ttf->pixelSize;
	item->anti_alias = ttf->antiAlias;
	item->cleartype = ttf->cleartype;
	item->embolden = ttf->embolden;
	item->xadvance = xadvance;
	item->yadvance = yadvance;
	item->bitmap.width = bitmap->width;
	item->bitmap.height = bitmap->rows;
	item->bitmap.left = left;
	item->bitmap.top = top;
	item->bitmap.pitch = bitmap->pitch;
	item->bitmap.format = bitmap->pixel_mode;
	item->bitmap.max_grays = (bitmap->num_grays - 1);

	int pitch = abs(bitmap->pitch);

	if (pitch * bitmap->rows > 0) {
		item->bitmap.buffer =
			(unsigned char *) memalign(64, pitch * bitmap->rows);
		if (item->bitmap.buffer) {
			memcpy(item->bitmap.buffer, bitmap->buffer, pitch * bitmap->rows);
		}
	}
}

/**
 * TTFλв
 */
static SBit_HashItem *sbitCacheFind(p_ttf ttf, unsigned long ucsCode)
{
	int i;

	for (i = 0; i < ttf->cacheSize; i++) {
		if ((ttf->sbitHashRoot[i].ucs_code == ucsCode) &&
			(ttf->sbitHashRoot[i].size == ttf->pixelSize) &&
			(ttf->sbitHashRoot[i].anti_alias == ttf->antiAlias) &&
			(ttf->sbitHashRoot[i].cleartype == ttf->cleartype) &&
			(ttf->sbitHashRoot[i].embolden == ttf->embolden))
			return (&ttf->sbitHashRoot[i]);
	}
	return NULL;
}

/**
 * 8λͨalpha㷨
 * @param wpSrc Դɫָ
 * @param wpDes Ŀɫָ
 * @param wAlpha alphaֵ(0-255)
 * @note Ŀɫ = Ŀɫ * alpha + ( 1 - alpha ) * Դɫ
 */
static __inline void MakeAlpha(byte * wpSrc, byte * wpDes, byte wAlpha)
{
	word result;
//	wAlpha = 0;
	if (*wpDes == *wpSrc && wAlpha == WALPHA_MAX)
		return;

	if (wAlpha == 0)
		*wpDes = *wpSrc;

	result = wAlpha * (*wpDes) + (WALPHA_MAX - wAlpha) * (*wpSrc);

	*wpDes = result / WALPHA_MAX;
}

/**
 * TTF͵Ļˮƽ汾
 * @param buffer λͼ
 * @param format ʽ
 * @param width λͼ
 * @param height λͼ߶
 * @param pitch λͼλֽڼǣ
 * @param x Ļx
 * @param y Ļy
 * @param scr_width ͼ
 * @param scr_height ͼ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void _drawBitmap_horz(byte * buffer, int format, int width, int height,
							 int pitch, FT_Int x, FT_Int y,
							 int scr_width, int scr_height, pixel color)
{
	FT_Int i, j;
	pixel pix, grey;
	byte ra, ga, ba;
	byte rd, gd, bd;
	byte rs, gs, bs;

	if (!buffer)
		return;
//  dbg_printf(d, "%s: %d, %d, %d", __func__, x, y, scr_height);

//------------------------------------------
#if 1
	switch( format )
	{
		case 0:dprintf( "FT_PIXEL_MODE_NONE\n" );break;
		case 1:dprintf( "FT_PIXEL_MODE_MONO\n" );break;
		case 2:dprintf( "FT_PIXEL_MODE_GRAY\n" );break;
		case 3:dprintf( "FT_PIXEL_MODE_GRAY2\n" );break;
		case 4:dprintf( "FT_PIXEL_MODE_GRAY4\n" );break;
		case 5:dprintf( "FT_PIXEL_MODE_LCD\n" );break;
		case 6:dprintf( "FT_PIXEL_MODE_LCD_V\n" );break;
	}
#endif
//	format = FT_PIXEL_MODE_GRAY;
	if (format == FT_PIXEL_MODE_MONO) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (i + x < 0 || i + x >= scr_width || j + y < 0
					|| j + y >= scr_height)
					continue;
				if (buffer[j * pitch + i / 8] & (0x80 >> (i % 8)))
					*(pixel *) disp_get_vaddr((i + x), (j + y)) = (color);
			}
	} else if (format == FT_PIXEL_MODE_GRAY) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (i + x < 0 || i + x >= scr_width || j + y < 0
					|| j + y >= scr_height)
					continue;
				grey = buffer[j * pitch + i];

				if (grey) {
					pix =
						disp_grayscale(*disp_get_vaddr((i + x), (j + y)),
									   RGB_R(color), RGB_G(color), RGB_B(color),
									   grey * 100 / 255);
					*(pixel *) disp_get_vaddr((i + x), (j + y)) = (pix);
				}
			}
	} else if (format == FT_PIXEL_MODE_LCD) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width / 3; i++) {
				if (i + x < 0 || i + x >= scr_width || j + y < 0 || j + y >= scr_height)
					continue;
				// RGB or BGR ?
				pixel origcolor = *disp_get_vaddr((i + x), (j + y));

				ra = buffer[j * pitch + i * 3];
				ga = buffer[j * pitch + i * 3 + 1];
				ba = buffer[j * pitch + i * 3 + 2];

//				dprintf( "origcolor = %04X\n", origcolor );
				dprintf( "%02X %02X %02X\n\n", ra, ga, ba );

				if( ra == 0 && ga == 0 && ba == 0 )
					continue;
//				else
//					ra = ga = ba = 255;

				rs = RGB_R(origcolor);
				gs = RGB_G(origcolor);
				bs = RGB_B(origcolor);

				rd = RGB_R(color);
				gd = RGB_G(color);
				bd = RGB_B(color);

#define			USE_RGB15		0

#if USE_RGB15
				ra = ra * 31 / 255;
				ga = ga * 31 / 255;
				ba = ba * 31 / 255;
#else
				dprintf( "rs gs bs: %02X %02X %02X\n", rs, gs, bs );
				rs = rs*255/31;
				gs = gs*255/31;
				bs = bs*255/31;
				dprintf( "rs gs bs: %02X %02X %02X\n\n", rs, gs, bs );

				dprintf( "rd gd bd: %02X %02X %02X\n", rd, gd, bd );
				rd = rd*255/31;
				gd = gd*255/31;
				bd = bd*255/31;
				dprintf( "[0]rd gd bd: %02X %02X %02X\n\n", rd, gd, bd );

#endif

				MakeAlpha(&rs, &rd, ra);
				MakeAlpha(&gs, &gd, ga);
				MakeAlpha(&bs, &bd, ba);

				dprintf( "[1]rd gd bd: %02X %02X %02X\n\n", rd, gd, bd );

#if USE_RGB15
//				*(pixel *) disp_get_vaddr((i + x), (j + y)) = color;
				*(pixel *) disp_get_vaddr((i + x), (j + y)) = (RGB15(rd, gd, bd));
#else
				*(pixel *) disp_get_vaddr((i + x), (j + y)) = (RGB(rd, gd, bd));
#endif
			}
	}
}

/**
 * TTFͻ浽Ļˮƽ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawCachedBitmap_horz(Cache_Bitmap * sbt, FT_Int x, FT_Int y,
								  int width, int height, pixel color)
{
	_drawBitmap_horz(sbt->buffer, sbt->format, sbt->width, sbt->height,
					 sbt->pitch, x, y, width, height, color);
}

/**
 * TTF͵Ļˮƽ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawBitmap_horz(FT_Bitmap * bitmap, FT_Int x, FT_Int y,
							int width, int height, pixel color)
{
	if (!bitmap->buffer)
		return;

	_drawBitmap_horz(bitmap->buffer, bitmap->pixel_mode, bitmap->width,
					 bitmap->rows, bitmap->pitch, x, y, width, height, color);
}

#define DISP_RSPAN 0

/**
 * *strȡһ֣/Ӣĸлƣˮƽ汾
 * @param ttf ʹõTTF
 * @param *x ָXָ
 * @param *y ָYָ
 * @param color ɫ
 * @param str ַָָ
 * @param count ָ
 * @param wordspace ּࣨصƣ
 * @param top ĿʼУͲĻ⣬Ϊ0ֵ
 * @param height ĸ߶ȣβĻ⣬
 * 				 򲻵DISP_BOOK_FONTSIZEڱüĸ߶
 * @param bot Ƹ߶
 * @param previous ָһַָͬ
 * @param is_hanzi ǷΪ
 * @note һͣ*x, *y, *count, *strᱻ
 * <br> ڻƵһУbot = 0ڻУbot = Ƹ߶
 */
static void ttf_disp_putnstring_horz(p_ttf ttf, int *x, int *y, pixel color,
									 const byte ** str, int *count,
									 dword wordspace, int top, int height,
									 int bot, FT_UInt * previous, BOOL is_hanzi)
{
	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;

	FT_Bool useKerning;

	useKerning = FT_HAS_KERNING(ttf->face);
	word ucs = charsets_gbk_to_ucs(*str);
	SBit_HashItem *cache = sbitCacheFind(ttf, ucs);

	if (cache) {
		if (useKerning && *previous && cache->glyph_index) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, cache->glyph_index,
						   FT_KERNING_DEFAULT, &delta);
			*x += delta.x >> 6;
		}
		drawCachedBitmap_horz(&cache->bitmap, *x + cache->bitmap.left,
							  bot ? *y + DISP_BOOK_FONTSIZE -
							  cache->bitmap.top : *y + height -
							  cache->bitmap.top,
							  SCREEN_W, bot ? bot : SCREEN_H,
							  color);
		*x += cache->xadvance >> 6;
		*previous = cache->glyph_index;
	} else {
		glyphIndex = FT_Get_Char_Index(ttf->face, ucs);
		// disable hinting when loading chinese characters
		if (is_hanzi)
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_NO_HINTING);
		else
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_DEFAULT);
		if (error)
			return;
		if (ttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
			if (ttf->cleartype) {
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_LCD);
			} else if (ttf->antiAlias) {
				error =
					FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_NORMAL);
			} else
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_MONO);
			if (error) {
				return;
			}
		}
		slot = ttf->face->glyph;

		if (ttf->embolden)
			FT_GlyphSlot_Embolden(slot);

		if (useKerning && *previous && glyphIndex) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, glyphIndex, FT_KERNING_DEFAULT,
						   &delta);
			*x += delta.x >> 6;
		}
		drawBitmap_horz(&slot->bitmap, *x + slot->bitmap_left,
						bot ? *y + DISP_BOOK_FONTSIZE - slot->bitmap_top : *y +
						height - slot->bitmap_top, SCREEN_W,
						bot ? bot : SCREEN_H, color);
		*x += slot->advance.x >> 6;
		*previous = glyphIndex;

		sbitCacheAdd(ttf, ucs, glyphIndex,
					 &slot->bitmap, slot->bitmap_left, slot->bitmap_top,
					 slot->advance.x, slot->advance.y);
	}
	if (is_hanzi) {
		(*str) += 2;
		*count -= 2;
		*x += wordspace * 2;
	} else {
		(*str)++;
		(*count)--;
		*x += wordspace;
	}
}

/**
 * õַʾmaxpixelsеĳ
 * @param cttf TTF
 * @param ettf ӢTTF
 * @param str ַ
 * @param maxpixels 󳤶
 * @param wordspace ּࣨصƣ
 * @note Уַֹͣۼӡ
 * <br>  ַƳȣmaxpixelsַֹͣۼӡ
 * <br>  汾ٶȼҪʹ
 */
extern int ttf_get_string_width_hard(p_ttf cttf, p_ttf ettf, const byte * str,
									 dword maxpixels, dword wordspace)
{
	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;
	FT_Bool useKerning;
	FT_UInt cprevious = 0, eprevious = 0;
	int x = 0, count = 0;

	if (str == NULL || maxpixels == 0 || cttf == NULL || ettf == NULL)
		return 0;

	while (*str != 0 && x < maxpixels && bytetable[*(byte *) str] != 1) {
		if (*str > 0x80) {
			useKerning = FT_HAS_KERNING(cttf->face);
			word ucs = charsets_gbk_to_ucs(str);
			SBit_HashItem *cache = sbitCacheFind(cttf, ucs);

			if (cache) {
				if (useKerning && cprevious && cache->glyph_index) {
					FT_Vector delta;

					FT_Get_Kerning(cttf->face, cprevious, cache->glyph_index,
								   FT_KERNING_DEFAULT, &delta);
					x += delta.x >> 6;
					if (x > maxpixels)
						break;
				}
				x += cache->xadvance >> 6;
				if (x > maxpixels)
					break;
				cprevious = cache->glyph_index;
			} else {
				glyphIndex = FT_Get_Char_Index(cttf->face, ucs);
				// disable hinting when loading chinese characters
				error =
					FT_Load_Glyph(cttf->face, glyphIndex, FT_LOAD_NO_HINTING);
				if (error)
					return count;
				if (cttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
					if (cttf->cleartype) {
						error =
							FT_Render_Glyph(cttf->face->glyph,
											FT_RENDER_MODE_LCD);
					} else if (cttf->antiAlias) {
						error =
							FT_Render_Glyph(cttf->face->glyph,
											FT_RENDER_MODE_NORMAL);
					} else
						error =
							FT_Render_Glyph(cttf->face->glyph,
											FT_RENDER_MODE_MONO);
					if (error) {
						return count;
					}
				}
				slot = cttf->face->glyph;

				if (cttf->embolden)
					FT_GlyphSlot_Embolden(slot);

				if (useKerning && cprevious && glyphIndex) {
					FT_Vector delta;

					FT_Get_Kerning(cttf->face, cprevious, glyphIndex,
								   FT_KERNING_DEFAULT, &delta);
					x += delta.x >> 6;
					if (x > maxpixels)
						break;
				}
				x += slot->advance.x >> 6;
				if (x > maxpixels)
					break;
				cprevious = glyphIndex;

				sbitCacheAdd(cttf, ucs, glyphIndex,
							 &slot->bitmap, slot->bitmap_left, slot->bitmap_top,
							 slot->advance.x, slot->advance.y);
			}
			if (x > maxpixels)
				break;
			x += wordspace * 2;
			(str) += 2;
			(count) += 2;
		} else if (*str > 0x1F) {
			useKerning = FT_HAS_KERNING(ettf->face);
			word ucs = charsets_gbk_to_ucs(str);
			SBit_HashItem *cache = sbitCacheFind(ettf, ucs);

			if (cache) {
				if (useKerning && eprevious && cache->glyph_index) {
					FT_Vector delta;

					FT_Get_Kerning(ettf->face, eprevious, cache->glyph_index,
								   FT_KERNING_DEFAULT, &delta);
					x += delta.x >> 6;
					if (x > maxpixels)
						break;
				}
				x += cache->xadvance >> 6;
				if (x > maxpixels)
					break;
				eprevious = cache->glyph_index;
			} else {
				glyphIndex = FT_Get_Char_Index(ettf->face, ucs);
				// disable hinting when loading chinese characters
				error = FT_Load_Glyph(ettf->face, glyphIndex, FT_LOAD_DEFAULT);
				if (error)
					return count;
				if (ettf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
					if (ettf->cleartype) {
						error =
							FT_Render_Glyph(ettf->face->glyph,
											FT_RENDER_MODE_LCD);
					} else if (ettf->antiAlias) {
						error =
							FT_Render_Glyph(ettf->face->glyph,
											FT_RENDER_MODE_NORMAL);
					} else
						error =
							FT_Render_Glyph(ettf->face->glyph,
											FT_RENDER_MODE_MONO);
					if (error) {
						return count;
					}
				}
				slot = ettf->face->glyph;

				if (ettf->embolden)
					FT_GlyphSlot_Embolden(slot);

				if (useKerning && eprevious && glyphIndex) {
					FT_Vector delta;

					FT_Get_Kerning(ettf->face, eprevious, glyphIndex,
								   FT_KERNING_DEFAULT, &delta);
					x += delta.x >> 6;
					if (x > maxpixels)
						break;
				}
				x += slot->advance.x >> 6;
				if (x > maxpixels)
					break;
				eprevious = glyphIndex;

				sbitCacheAdd(ettf, ucs, glyphIndex,
							 &slot->bitmap, slot->bitmap_left, slot->bitmap_top,
							 slot->advance.x, slot->advance.y);
			}
			if (x > maxpixels)
				break;
			x += wordspace;
			(str)++;
			(count)++;
		} else {
			int j;

			for (j = 0; j < (*str == 0x09 ? config.tabstop : 1); ++j)
				x += DISP_BOOK_FONTSIZE / 2 + wordspace;
			if (x > maxpixels)
				break;
			str++;
			(count)++;
		}
	}
	if (bytetable[*(byte *) str] == 1) {
		if (*str == '\r' && *(str + 1) == '\n') {
			count += 2;
		} else {
			count++;
		}
	}

	return count;
}

static int ttf_get_char_width(p_ttf cttf, const byte * str)
{
	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;
	FT_Bool useKerning;
	int x = 0;

	if (str == NULL)
		return 0;
	if (cttf == NULL)
		return DISP_BOOK_FONTSIZE;

	useKerning = FT_HAS_KERNING(cttf->face);
	word ucs = charsets_gbk_to_ucs(str);

	glyphIndex = FT_Get_Char_Index(cttf->face, ucs);
	// disable hinting when loading chinese characters
	error = FT_Load_Glyph(cttf->face, glyphIndex, FT_LOAD_NO_HINTING);
	if (error)
		return x;
	if (cttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
		if (cttf->cleartype) {
			error = FT_Render_Glyph(cttf->face->glyph, FT_RENDER_MODE_LCD);
		} else if (cttf->antiAlias) {
			error = FT_Render_Glyph(cttf->face->glyph, FT_RENDER_MODE_NORMAL);
		} else
			error = FT_Render_Glyph(cttf->face->glyph, FT_RENDER_MODE_MONO);
		if (error) {
			return x;
		}
	}
	slot = cttf->face->glyph;

	if (cttf->embolden)
		FT_GlyphSlot_Embolden(slot);

	x += slot->advance.x >> 6;

	return x;
}

/**
 * õַʾmaxpixelsеĳ
 * @param cttf TTF
 * @param ettf ӢTTF
 * @param str ַ
 * @param maxpixels 󳤶
 * @param wordspace ּࣨصƣ
 * @return ַֽڼ
 * @note Уַֹͣۼӡ
 * <br>  ַƳȣmaxpixelsַֹͣۼӡ
 * <br>  汾ٶȿ죬Ӣĸг
 */
extern int ttf_get_string_width(p_ttf cttf, p_ttf ettf, const byte * str,
								dword maxpixels, dword wordspace)
{
	dword width = 0;
	const byte *ostr = str;
	static int hanzi_len, hanzi_size = 0;

	if (hanzi_len == 0 || hanzi_size != DISP_BOOK_FONTSIZE) {
		hanzi_len = ttf_get_char_width(cttf, (const byte *) "");
		hanzi_size = DISP_BOOK_FONTSIZE;
	}

	while (*str != 0 && width <= maxpixels && bytetable[*str] != 1) {
		if (*str > 0x80) {
			width += hanzi_len;
			if (width > maxpixels)
				break;
			width += wordspace * 2;
			str += 2;
		} else if (*str == 0x20) {
			width += DISP_BOOK_FONTSIZE / 2;
			if (width > maxpixels)
				break;
			width += wordspace;
			str++;
		} else {
			if (*str == 0x09) {
				int j;

				for (j = 0; j < config.tabstop; ++j)
					width += DISP_BOOK_FONTSIZE / 2;
			} else
				width += disp_ewidth[*str];
			if (width > maxpixels)
				break;
			width += wordspace;
			str++;
		}
	}

	return str - ostr;
}

extern void disp_putnstring_horz_truetype2(p_ttf cttf, p_ttf ettf, int x, int y,
										  pixel color, const byte * str,
										  int count, dword wordspace, int top,
										  int height, int bot)
{
	if (cttf == NULL || ettf == NULL)
		return;

	if (bot) {
		if (y >= bot)
			return;
		if (y + height > bot)
			height = bot - y;
//      dbg_printf(d, "%s: bot now: height: %d bot %d", __func__, height, bot);
	}

	/*
	   #define CHECK_AND_VALID(x, y) \
	   {\
	   x = (x < 0) ? 0 : x; \
	   y = (y < 0) ? 0 : y; \
	   x = (x >= SCREEN_W )? SCREEN_W - 1: x;\
	   y = (y >= SCREEN_H )? SCREEN_H - 1: y;\
	   }

	   CHECK_AND_VALID(x, y);
	 */

	FT_UInt cprevious, eprevious;

	cprevious = eprevious = 0;

	while (*str != 0 && count > 0) {
		if (!check_range(x, y))
			return;
		if (*str > 0x80) {
			ttf_disp_putnstring_horz(cttf, &x, &y, color, &str, &count,
									 wordspace, top, height, bot,
									 &cprevious, true);
		} else if (*str > 0x1F && *str != 0x20) {
			ttf_disp_putnstring_horz(ettf, &x, &y, color, &str, &count,
									 wordspace, top, height, bot,
									 &eprevious, false);
		} else {
			if (x > SCREEN_W - DISP_RSPAN - DISP_BOOK_FONTSIZE / 2) {
				break;
#if 0
				x = 0;
				y += DISP_BOOK_FONTSIZE;
#endif
			}
			int j;

			for (j = 0; j < (*str == 0x09 ? config.tabstop : 1); ++j)
				x += DISP_BOOK_FONTSIZE / 2 + wordspace;
			str++;
			count--;
		}
	}
}


extern void disp_putnstring_horz_truetype( int x, int y,
										  pixel color, const byte * str,
										  int count, dword wordspace, int top,
										  int height, int bot)
{
	disp_putnstring_horz_truetype2( cttf, ettf, x, y, color, str, count, wordspace, top, height, bot );
}


/**
 * õӢĸϢ
 */
extern void ttf_load_ewidth(p_ttf ttf, byte * ewidth, int size)
{
	if (ttf == NULL || ewidth == NULL || size == 0)
		return;

	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;
	FT_Bool useKerning;
	FT_UInt eprevious = 0;
	byte width;
	word ucs;

	for (ucs = 0; ucs < size; ++ucs) {
		width = 0;
		useKerning = FT_HAS_KERNING(ttf->face);
		glyphIndex = FT_Get_Char_Index(ttf->face, ucs);
		error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_DEFAULT);
		if (error)
			return;
		if (ttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
			if (ttf->cleartype) {
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_LCD);
			} else if (ttf->antiAlias) {
				error =
					FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_NORMAL);
			} else
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_MONO);
			if (error) {
				return;
			}
		}
		slot = ttf->face->glyph;

		if (ttf->embolden)
			FT_GlyphSlot_Embolden(slot);

		if (useKerning && eprevious && glyphIndex) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, eprevious, glyphIndex,
						   FT_KERNING_DEFAULT, &delta);
			width += delta.x >> 6;
		}
		width += slot->advance.x >> 6;
		eprevious = glyphIndex;

		// Add width to length
		ewidth[ucs] = width;
	}

	F( "%s: OK. 'i' width is %d, 'M' width is %d\n\n", __func__, ewidth['i'], ewidth['M']);
}

/**
 * TTF͵Ļߵ汾
 * @param buffer λͼ
 * @param format ʽ
 * @param width λͼ
 * @param height λͼ߶
 * @param pitch λͼλֽڼǣ
 * @param x Ļx
 * @param y Ļy
 * @param scr_width ͼ
 * @param scr_height ͼ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void _drawBitmap_reversal(byte * buffer, int format, int width,
								 int height, int pitch, FT_Int x, FT_Int y,
								 int scr_width, int scr_height, pixel color)
{
	FT_Int i, j;
	pixel pix, grey;
	byte ra, ga, ba;
	byte rd, gd, bd;
	byte rs, gs, bs;

	if (!buffer)
		return;
	if (format == FT_PIXEL_MODE_MONO) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (x - i < 0 || x - i >= scr_width
					|| y - j < SCREEN_H - scr_height
					|| y - j >= SCREEN_H)
					continue;
				if (buffer[j * pitch + i / 8] & (0x80 >> (i % 8)))
					*(pixel *) disp_get_vaddr((x - i), (y - j)) = (color);
			}
	} else if (format == FT_PIXEL_MODE_GRAY) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (x - i < 0 || x - i >= scr_width
					|| y - j < SCREEN_H - scr_height
					|| y - j >= SCREEN_H)
					continue;
				grey = buffer[j * pitch + i];

				if (grey) {
					pix =
						disp_grayscale(*disp_get_vaddr((x - i), (y - j)),
									   RGB_R(color), RGB_G(color), RGB_B(color),
									   grey * 100 / 255);
					*(pixel *) disp_get_vaddr((x - i), (y - j)) = (pix);
				}
			}
	} else if (format == FT_PIXEL_MODE_LCD) {
//      dbg_printf(d, "%s: %d, %d, %d", __func__, x, y, scr_height);
		for (j = 0; j < height; j++)
			for (i = 0; i < width / 3; i++) {
				if (x - i < 0 || x - i >= scr_width
					|| y - j < SCREEN_H - scr_height
					|| y - j >= SCREEN_H)
					continue;
				// RGB or BGR ?
				pixel origcolor = *disp_get_vaddr((x - i), (y - j));

				ra = buffer[j * pitch + i * 3];
				ga = buffer[j * pitch + i * 3 + 1];
				ba = buffer[j * pitch + i * 3 + 2];

				rs = RGB_R(origcolor);
				gs = RGB_G(origcolor);
				bs = RGB_B(origcolor);

				rd = RGB_R(color);
				gd = RGB_G(color);
				bd = RGB_B(color);

				MakeAlpha(&rs, &rd, ra);
				MakeAlpha(&gs, &gd, ga);
				MakeAlpha(&bs, &bd, ba);

				*(pixel *) disp_get_vaddr((x - i), (y - j)) = (RGB(rd, gd, bd));
			}
	}
}

/**
 * TTFͻ浽Ļߵ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawCachedBitmap_reversal(Cache_Bitmap * sbt, FT_Int x, FT_Int y,
									  int width, int height, pixel color)
{
	_drawBitmap_reversal(sbt->buffer, sbt->format, sbt->width, sbt->height,
						 sbt->pitch, x, y, width, height, color);
}

/**
 * TTF͵Ļߵ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawBitmap_reversal(FT_Bitmap * bitmap, FT_Int x, FT_Int y,
								int width, int height, pixel color)
{
	if (!bitmap->buffer)
		return;

	_drawBitmap_reversal(bitmap->buffer, bitmap->pixel_mode, bitmap->width,
						 bitmap->rows, bitmap->pitch, x, y, width, height,
						 color);
}

/**
 * *strȡһ֣/Ӣĸлƣߵ汾
 * @param ttf ʹõTTF
 * @param *x ָXָ
 * @param *y ָYָ
 * @param color ɫ
 * @param str ַָָ
 * @param count ָ
 * @param wordspace ּࣨصƣ
 * @param top ĿʼУͲĻ⣬Ϊ0ֵ
 * @param height ĸ߶ȣβĻ⣬
 * 				 򲻵DISP_BOOK_FONTSIZEڱüĸ߶
 * @param bot Ƹ߶
 * @param previous ָһַָͬ
 * @param is_hanzi ǷΪ
 * @note һͣ*x, *y, *count, *strᱻ
 * <br> ڻƵһУbot = 0ڻУbot = Ƹ߶
 */
static void ttf_disp_putnstring_reversal(p_ttf ttf, int *x, int *y, pixel color,
										 const byte ** str, int *count,
										 dword wordspace, int top, int height,
										 int bot, FT_UInt * previous,
										 BOOL is_hanzi)
{
	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;

	FT_Bool useKerning;

	useKerning = FT_HAS_KERNING(ttf->face);
	word ucs = charsets_gbk_to_ucs(*str);
	SBit_HashItem *cache = sbitCacheFind(ttf, ucs);

	if (cache) {
		if (useKerning && *previous && cache->glyph_index) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, cache->glyph_index,
						   FT_KERNING_DEFAULT, &delta);
			*x -= delta.x >> 6;
		}
		drawCachedBitmap_reversal(&cache->bitmap, *x - cache->bitmap.left,
								  bot ? *y - DISP_BOOK_FONTSIZE +
								  cache->bitmap.top : *y - height +
								  cache->bitmap.top,
								  SCREEN_W,
								  bot ? bot : SCREEN_H, color);
		*x -= cache->xadvance >> 6;
		*previous = cache->glyph_index;
	} else {
		glyphIndex = FT_Get_Char_Index(ttf->face, ucs);
		// disable hinting when loading chinese characters
		if (is_hanzi)
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_NO_HINTING);
		else
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_DEFAULT);
		if (error)
			return;
		if (ttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
			if (ttf->cleartype) {
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_LCD);
			} else if (ttf->antiAlias) {
				error =
					FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_NORMAL);
			} else
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_MONO);
			if (error) {
				return;
			}
		}
		slot = ttf->face->glyph;

		if (ttf->embolden)
			FT_GlyphSlot_Embolden(slot);

		if (useKerning && *previous && glyphIndex) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, glyphIndex, FT_KERNING_DEFAULT,
						   &delta);
			*x -= delta.x >> 6;
		}
		drawBitmap_reversal(&slot->bitmap, *x - slot->bitmap_left,
							bot ? *y - DISP_BOOK_FONTSIZE +
							slot->bitmap_top : *y - height + slot->bitmap_top,
							SCREEN_W, bot ? bot : SCREEN_H,
							color);
		*x -= slot->advance.x >> 6;
		*previous = glyphIndex;

		sbitCacheAdd(ttf, ucs, glyphIndex,
					 &slot->bitmap, slot->bitmap_left, slot->bitmap_top,
					 slot->advance.x, slot->advance.y);
	}
	if (is_hanzi) {
		(*str) += 2;
		*count -= 2;
		*x -= wordspace * 2;
	} else {
		(*str)++;
		(*count)--;
		*x -= wordspace;
	}
}

extern void disp_putnstring_reversal_truetype(p_ttf cttf, p_ttf ettf, int x,
											  int y, pixel color,
											  const byte * str, int count,
											  dword wordspace, int top,
											  int height, int bot)
{
	if (cttf == NULL || ettf == NULL)
		return;

	if (bot) {
		if (y >= bot)
			return;
		if (y + height > bot)
			height = bot - y;
//      dbg_printf(d, "%s: bot now: height: %d bot %d", __func__, height, bot);
	}

	/*
	   #define CHECK_AND_VALID(x, y) \
	   {\
	   x = (x < 0) ? 0 : x; \
	   y = (y < 0) ? 0 : y; \
	   x = (x >= SCREEN_W )? SCREEN_W - 1: x;\
	   y = (y >= SCREEN_H )? SCREEN_H - 1: y;\
	   }

	   CHECK_AND_VALID(x, y);
	 */

	x = SCREEN_W - x - 1, y = SCREEN_H - y - 1;

	FT_UInt cprevious, eprevious;

	cprevious = eprevious = 0;

	while (*str != 0 && count > 0) {
		if (!check_range(x, y))
			return;
		if (x < 0)
			break;
		if (*str > 0x80) {
			ttf_disp_putnstring_reversal(cttf, &x, &y, color, &str, &count,
										 wordspace, top, height, bot,
										 &cprevious, true);
		} else if (*str > 0x1F && *str != 0x20) {
			ttf_disp_putnstring_reversal(ettf, &x, &y, color, &str, &count,
										 wordspace, top, height, bot,
										 &eprevious, false);
		} else {
			if (x < 0) {
				break;
#if 0
				x = 0;
				y += DISP_BOOK_FONTSIZE;
#endif
			}
			int j;

			for (j = 0; j < (*str == 0x09 ? config.tabstop : 1); ++j)
				x -= DISP_BOOK_FONTSIZE / 2 + wordspace;
			str++;
			count--;
		}
	}
}

/**
 * TTF͵Ļ汾
 * @param buffer λͼ
 * @param format ʽ
 * @param width λͼ
 * @param height λͼ߶
 * @param pitch λͼλֽڼǣ
 * @param x Ļx
 * @param y Ļy
 * @param scr_width ͼ
 * @param scr_height ͼ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void _drawBitmap_lvert(byte * buffer, int format, int width, int height,
							  int pitch, FT_Int x, FT_Int y, int scr_width,
							  int scr_height, pixel color)
{
	FT_Int i, j;
	pixel pix, grey;
	byte ra, ga, ba;
	byte rd, gd, bd;
	byte rs, gs, bs;

	if (!buffer)
		return;
	if (format == FT_PIXEL_MODE_MONO) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (y - i < 0 || y - i >= scr_height || x + j < 0
					|| x + j >= scr_width)
					continue;
				if (buffer[j * pitch + i / 8] & (0x80 >> (i % 8)))
					*(pixel *) disp_get_vaddr((x + j), (y - i)) = (color);
			}
	} else if (format == FT_PIXEL_MODE_GRAY) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (y - i < 0 || y - i >= scr_height || x + j < 0
					|| x + j >= scr_width)
					continue;
				grey = buffer[j * pitch + i];

				if (grey) {
					pix =
						disp_grayscale(*disp_get_vaddr((x + j), (y - i)),
									   RGB_R(color), RGB_G(color), RGB_B(color),
									   grey * 100 / 255);
					*(pixel *) disp_get_vaddr((x + j), (y - i)) = (pix);
				}
			}
	} else if (format == FT_PIXEL_MODE_LCD) {
//      dbg_printf(d, "%s: %d, %d, %d", __func__, x, y, scr_height);
		for (j = 0; j < height; j++)
			for (i = 0; i < width / 3; i++) {
				if (y - i < 0 || y - i >= scr_height || x + j < 0
					|| x + j >= scr_width)
					continue;
				// RGB or BGR ?
				pixel origcolor = *disp_get_vaddr((x + j), (y - i));

				ra = buffer[j * pitch + i * 3];
				ga = buffer[j * pitch + i * 3 + 1];
				ba = buffer[j * pitch + i * 3 + 2];

				rs = RGB_R(origcolor);
				gs = RGB_G(origcolor);
				bs = RGB_B(origcolor);

				rd = RGB_R(color);
				gd = RGB_G(color);
				bd = RGB_B(color);

				MakeAlpha(&rs, &rd, ra);
				MakeAlpha(&gs, &gd, ga);
				MakeAlpha(&bs, &bd, ba);

				*(pixel *) disp_get_vaddr((x + j), (y - i)) = (RGB(rd, gd, bd));
			}
	}
}

/**
 * TTFͻ浽Ļ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawCachedBitmap_lvert(Cache_Bitmap * sbt, FT_Int x, FT_Int y,
								   int width, int height, pixel color)
{
	_drawBitmap_lvert(sbt->buffer, sbt->format, sbt->width, sbt->height,
					  sbt->pitch, x, y, width, height, color);
}

/**
 * TTF͵Ļ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawBitmap_lvert(FT_Bitmap * bitmap, FT_Int x, FT_Int y,
							 int width, int height, pixel color)
{
	if (!bitmap->buffer)
		return;

	_drawBitmap_lvert(bitmap->buffer, bitmap->pixel_mode, bitmap->width,
					  bitmap->rows, bitmap->pitch, x, y, width, height, color);
}

/**
 * *strȡһ֣/Ӣĸлƣ汾
 * @param ttf ʹõTTF
 * @param *x ָXָ
 * @param *y ָYָ
 * @param color ɫ
 * @param str ַָָ
 * @param count ָ
 * @param wordspace ּࣨصƣ
 * @param top ĿʼУͲĻ⣬Ϊ0ֵ
 * @param height ĸ߶ȣβĻ⣬
 * 				 򲻵DISP_BOOK_FONTSIZEڱüĸ߶
 * @param bot Ƹ߶
 * @param previous ָһַָͬ
 * @param is_hanzi ǷΪ
 * @note һͣ*x, *y, *count, *strᱻ
 * <br> ڻƵһУbot = 0ڻУbot = Ƹ߶
 */
static void ttf_disp_putnstring_lvert(p_ttf ttf, int *x, int *y, pixel color,
									  const byte ** str, int *count,
									  dword wordspace, int top, int height,
									  int bot, FT_UInt * previous,
									  BOOL is_hanzi)
{
	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;

	FT_Bool useKerning;

	useKerning = FT_HAS_KERNING(ttf->face);
	word ucs = charsets_gbk_to_ucs(*str);
	SBit_HashItem *cache = sbitCacheFind(ttf, ucs);

	if (cache) {
		if (useKerning && *previous && cache->glyph_index) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, cache->glyph_index,
						   FT_KERNING_DEFAULT, &delta);
			*y -= delta.x >> 6;
		}
		drawCachedBitmap_lvert(&cache->bitmap,
							   bot ? *x + DISP_BOOK_FONTSIZE -
							   cache->bitmap.top : *x + height -
							   cache->bitmap.top, *y - cache->bitmap.left,
							   bot ? bot : SCREEN_W, SCREEN_H,
							   color);
		*y -= cache->xadvance >> 6;
		*previous = cache->glyph_index;
	} else {
		glyphIndex = FT_Get_Char_Index(ttf->face, ucs);
		// disable hinting when loading chinese characters
		if (is_hanzi)
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_NO_HINTING);
		else
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_DEFAULT);
		if (error)
			return;
		if (ttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
			if (ttf->cleartype) {
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_LCD);
			} else if (ttf->antiAlias) {
				error =
					FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_NORMAL);
			} else
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_MONO);
			if (error) {
				return;
			}
		}
		slot = ttf->face->glyph;

		if (ttf->embolden)
			FT_GlyphSlot_Embolden(slot);

		if (useKerning && *previous && glyphIndex) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, glyphIndex, FT_KERNING_DEFAULT,
						   &delta);
			*y -= delta.x >> 6;
		}
		drawBitmap_lvert(&slot->bitmap,
						 bot ? *x + DISP_BOOK_FONTSIZE - slot->bitmap_top : *x +
						 height - slot->bitmap_top, *y - slot->bitmap_left,
						 bot ? bot : SCREEN_W, SCREEN_H,
						 color);
		*y -= slot->advance.x >> 6;
		*previous = glyphIndex;

		sbitCacheAdd(ttf, ucs, glyphIndex,
					 &slot->bitmap, slot->bitmap_left, slot->bitmap_top,
					 slot->advance.x, slot->advance.y);
	}
	if (is_hanzi) {
		(*str) += 2;
		*count -= 2;
		*y -= wordspace * 2;
	} else {
		(*str)++;
		(*count)--;
		*y -= wordspace;
	}
}

extern void disp_putnstring_lvert_truetype(p_ttf cttf, p_ttf ettf, int x, int y,
										   pixel color, const byte * str,
										   int count, dword wordspace, int top,
										   int height, int bot)
{
	if (cttf == NULL || ettf == NULL)
		return;

	if (bot) {
		if (x >= bot)
			return;
		if (x + height > bot)
			height = bot - x;
//      dbg_printf(d, "%s: bot now: height: %d bot %d", __func__, height, bot);
	}

	/*
	   #define CHECK_AND_VALID(x, y) \
	   {\
	   x = (x < 0) ? 0 : x; \
	   y = (y < 0) ? 0 : y; \
	   x = (x >= SCREEN_W )? SCREEN_W - 1: x;\
	   y = (y >= SCREEN_H )? SCREEN_H - 1: y;\
	   }

	   CHECK_AND_VALID(x, y);
	 */

	FT_UInt cprevious, eprevious;

	cprevious = eprevious = 0;

	while (*str != 0 && count > 0) {
		if (!check_range(x, y))
			return;
		if (*str > 0x80) {
			if (y < DISP_RSPAN + DISP_BOOK_FONTSIZE - 1)
				break;
			ttf_disp_putnstring_lvert(cttf, &x, &y, color, &str, &count,
									  wordspace, top, height, bot,
									  &cprevious, true);
		} else if (*str > 0x1F && *str != 0x20) {
			if (y < DISP_RSPAN + disp_ewidth[*str] - 1)
				break;
			ttf_disp_putnstring_lvert(ettf, &x, &y, color, &str, &count,
									  wordspace, top, height, bot,
									  &eprevious, false);
		} else {
			if (y < DISP_RSPAN + DISP_BOOK_FONTSIZE - 1) {
				break;
			}
			int j;

			for (j = 0; j < (*str == 0x09 ? config.tabstop : 1); ++j)
				y -= DISP_BOOK_FONTSIZE / 2 + wordspace;
			str++;
			count--;
		}
	}
}

/**
 * TTF͵Ļ汾
 * @param buffer λͼ
 * @param format ʽ
 * @param width λͼ
 * @param height λͼ߶
 * @param pitch λͼλֽڼǣ
 * @param x Ļx
 * @param y Ļy
 * @param scr_width ͼ
 * @param scr_height ͼ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void _drawBitmap_rvert(byte * buffer, int format, int width, int height,
							  int pitch, FT_Int x, FT_Int y, int scr_width,
							  int scr_height, pixel color)
{
	FT_Int i, j;
	pixel pix, grey;
	byte ra, ga, ba;
	byte rd, gd, bd;
	byte rs, gs, bs;

	if (!buffer)
		return;
	if (format == FT_PIXEL_MODE_MONO) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (y + i < 0 || y + i >= scr_height
					|| x - j < SCREEN_W - scr_width
					|| x - j >= SCREEN_W)
					continue;
				if (buffer[j * pitch + i / 8] & (0x80 >> (i % 8)))
					*(pixel *) disp_get_vaddr((x - j), (y + i)) = (color);
			}
	} else if (format == FT_PIXEL_MODE_GRAY) {
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++) {
				if (y + i < 0 || y + i >= scr_height
					|| x - j < SCREEN_W - scr_width
					|| x - j >= SCREEN_W)
					continue;
				grey = buffer[j * pitch + i];

				if (grey) {
					pix =
						disp_grayscale(*disp_get_vaddr((x - j), (y + i)),
									   RGB_R(color), RGB_G(color), RGB_B(color),
									   grey * 100 / 255);
					*(pixel *) disp_get_vaddr((x - j), (y + i)) = (pix);
				}
			}
	} else if (format == FT_PIXEL_MODE_LCD) {
//      dbg_printf(d, "%s: %d, %d, %d", __func__, x, y, scr_height);
		for (j = 0; j < height; j++)
			for (i = 0; i < width / 3; i++) {
				if (y + i < 0 || y + i >= scr_height
					|| x - j < SCREEN_W - scr_width
					|| x - j >= SCREEN_W)
					continue;
				// RGB or BGR ?
				pixel origcolor = *disp_get_vaddr((x - j), (y + i));

				ra = buffer[j * pitch + i * 3];
				ga = buffer[j * pitch + i * 3 + 1];
				ba = buffer[j * pitch + i * 3 + 2];

				rs = RGB_R(origcolor);
				gs = RGB_G(origcolor);
				bs = RGB_B(origcolor);

				rd = RGB_R(color);
				gd = RGB_G(color);
				bd = RGB_B(color);

				MakeAlpha(&rs, &rd, ra);
				MakeAlpha(&gs, &gd, ga);
				MakeAlpha(&bs, &bd, ba);

				*(pixel *) disp_get_vaddr((x - j), (y + i)) = (RGB(rd, gd, bd));
			}
	}
}

/**
 * TTFͻ浽Ļ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawCachedBitmap_rvert(Cache_Bitmap * sbt, FT_Int x, FT_Int y,
								   int width, int height, pixel color)
{
	_drawBitmap_rvert(sbt->buffer, sbt->format, sbt->width, sbt->height,
					  sbt->pitch, x, y, width, height, color);
}

/**
 * TTF͵Ļ汾
 * @param bitmap λͼ
 * @param x Ļx
 * @param y Ļy
 * @param width 
 * @param height ߶
 * @param color ɫ
 * @note (x,y)ΪʾϽǣ"հ"
 */
static void drawBitmap_rvert(FT_Bitmap * bitmap, FT_Int x, FT_Int y,
							 int width, int height, pixel color, int space)
{
	if (!bitmap->buffer)
		return;

	_drawBitmap_rvert(bitmap->buffer, bitmap->pixel_mode, bitmap->width,
					  bitmap->rows, bitmap->pitch, x, y, width, height, color);
}

/**
 * *strȡһ֣/Ӣĸлƣ汾
 * @param ttf ʹõTTF
 * @param *x ָXָ
 * @param *y ָYָ
 * @param color ɫ
 * @param str ַָָ
 * @param count ָ
 * @param wordspace ּࣨصƣ
 * @param top ĿʼУͲĻ⣬Ϊ0ֵ
 * @param height ĸ߶ȣβĻ⣬
 * 				 򲻵DISP_BOOK_FONTSIZEڱüĸ߶
 * @param bot Ƹ߶
 * @param previous ָһַָͬ
 * @param is_hanzi ǷΪ
 * @note һͣ*x, *y, *count, *strᱻ
 * <br> ڻƵһУbot = 0ڻУbot = Ƹ߶
 */
static void ttf_disp_putnstring_rvert(p_ttf ttf, int *x, int *y, pixel color,
									  const byte ** str, int *count,
									  dword wordspace, int top, int height,
									  int bot, FT_UInt * previous,
									  BOOL is_hanzi)
{
	FT_Error error;
	FT_GlyphSlot slot;
	FT_UInt glyphIndex;

	FT_Bool useKerning;

	useKerning = FT_HAS_KERNING(ttf->face);
	word ucs = charsets_gbk_to_ucs(*str);
	SBit_HashItem *cache = sbitCacheFind(ttf, ucs);

	if (cache) {
		if (useKerning && *previous && cache->glyph_index) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, cache->glyph_index,
						   FT_KERNING_DEFAULT, &delta);
			*y += delta.x >> 6;
		}
		drawCachedBitmap_rvert(&cache->bitmap,
							   bot ? *x - DISP_BOOK_FONTSIZE +
							   cache->bitmap.top : *x - height +
							   cache->bitmap.top, *y + cache->bitmap.left,
							   bot ? SCREEN_W - bot : SCREEN_W,
							   SCREEN_H, color);
		*y += cache->xadvance >> 6;
		*previous = cache->glyph_index;
	} else {
		glyphIndex = FT_Get_Char_Index(ttf->face, ucs);
		// disable hinting when loading chinese characters
		if (is_hanzi)
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_NO_HINTING);
		else
			error = FT_Load_Glyph(ttf->face, glyphIndex, FT_LOAD_DEFAULT);
		if (error)
			return;
		if (ttf->face->glyph->format != FT_GLYPH_FORMAT_BITMAP) {
			if (ttf->cleartype) {
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_LCD);
			} else if (ttf->antiAlias) {
				error =
					FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_NORMAL);
			} else
				error = FT_Render_Glyph(ttf->face->glyph, FT_RENDER_MODE_MONO);
			if (error) {
				return;
			}
		}
		slot = ttf->face->glyph;

		if (ttf->embolden)
			FT_GlyphSlot_Embolden(slot);

		if (useKerning && *previous && glyphIndex) {
			FT_Vector delta;

			FT_Get_Kerning(ttf->face, *previous, glyphIndex, FT_KERNING_DEFAULT,
						   &delta);
			*y += delta.x >> 6;
		}
		drawBitmap_rvert(&slot->bitmap,
						 bot ? *x - DISP_BOOK_FONTSIZE + slot->bitmap_top : *x -
						 height + slot->bitmap_top, *y + slot->bitmap_left,
						 bot ? SCREEN_W - bot : SCREEN_W,
						 SCREEN_H, color,
						 DISP_BOOK_FONTSIZE - slot->bitmap_top);
		*y += slot->advance.x >> 6;
		*previous = glyphIndex;

		sbitCacheAdd(ttf, ucs, glyphIndex,
					 &slot->bitmap, slot->bitmap_left, slot->bitmap_top,
					 slot->advance.x, slot->advance.y);
	}
	if (is_hanzi) {
		(*str) += 2;
		*count -= 2;
		*y += wordspace * 2;
	} else {
		(*str)++;
		(*count)--;
		*y += wordspace;
	}
}

extern void disp_putnstring_rvert_truetype(p_ttf cttf, p_ttf ettf, int x, int y,
										   pixel color, const byte * str,
										   int count, dword wordspace, int top,
										   int height, int bot)
{
	if (cttf == NULL || ettf == NULL)
		return;

	if (x < bot)
		return;
	if (x + 1 - height < bot)
		height = x + 1 - bot;

	/*
	   #define CHECK_AND_VALID(x, y) \
	   {\
	   x = (x < 0) ? 0 : x; \
	   y = (y < 0) ? 0 : y; \
	   x = (x >= SCREEN_W )? SCREEN_W - 1: x;\
	   y = (y >= SCREEN_H )? SCREEN_H - 1: y;\
	   }

	   CHECK_AND_VALID(x, y);
	 */

	FT_UInt cprevious, eprevious;

	cprevious = eprevious = 0;

	while (*str != 0 && count > 0) {
		if (!check_range(x, y))
			return;
		if (*str > 0x80) {
			if (y > SCREEN_H - DISP_RSPAN - DISP_BOOK_FONTSIZE)
				break;
			ttf_disp_putnstring_rvert(cttf, &x, &y, color, &str, &count,
									  wordspace, top, height, bot,
									  &cprevious, true);
		} else if (*str > 0x1F && *str != 0x20) {
			if (y > SCREEN_H - DISP_RSPAN - disp_ewidth[*str])
				break;
			ttf_disp_putnstring_rvert(ettf, &x, &y, color, &str, &count,
									  wordspace, top, height, bot,
									  &eprevious, false);
		} else {
			if (y > SCREEN_H - DISP_RSPAN - DISP_BOOK_FONTSIZE / 2) {
				break;
			}
			int j;

			for (j = 0; j < (*str == 0x09 ? config.tabstop : 1); ++j)
				y += DISP_BOOK_FONTSIZE / 2 + wordspace;
			str++;
			count--;
		}
	}
}


#if 1

#include <display.h>
#include <strsafe.h>

extern int use_ttf;

#undef ENABLE_TTF //ʹttf

#ifdef ENABLE_TTF
static BOOL load_ttf_config(void)
{
	ttf_set_anti_alias(cttf, config.cfont_antialias);
	ttf_set_anti_alias(ettf, config.efont_antialias);
	ttf_set_cleartype(cttf, config.cfont_cleartype);
	ttf_set_cleartype(ettf, config.efont_cleartype);
	ttf_set_embolden(cttf, config.cfont_embolden);
	ttf_set_embolden(ettf, config.efont_embolden);

	return true;
}
#endif

extern BOOL disp_load_truetype_book_font(const char *ettffile,
										 const char *cttffile, int size)
{
#ifdef ENABLE_TTF
	use_ttf = 0;
	memset(disp_ewidth, size / 2, 0x80);
	if (book_efont_buffer != NULL && efont_buffer != book_efont_buffer) {
		free((void *) book_efont_buffer);
		book_efont_buffer = NULL;
	}
	if (book_localfont_buffer != NULL && localfont_buffer != book_localfont_buffer) {
		free((void *) book_localfont_buffer);
		book_localfont_buffer = NULL;
	}
	if (ettf == NULL) {
		if ((ettf = ttf_open(ettffile, size)) == NULL) {
			return false;
		}
	} else {
		ttf_set_pixel_size(ettf, size);
	}
	if (cttf == NULL) {
		if ((cttf = ttf_open(cttffile, size)) == NULL) {
			return false;
		}
	} else {
		ttf_set_pixel_size(cttf, size);
	}
	load_ttf_config();

	ttf_load_ewidth(ettf, disp_ewidth, 0x80);
	use_ttf = 1;
	return true;
#else
	return false;
#endif
}

#ifdef ENABLE_TTF
static p_ttf load_archieve_truetype_book_font(const char *zipfile,
											  const char *zippath, int size)
{
	p_ttf ttf = NULL;

	if (ttf == NULL && zipfile[0] != '\0') {
		buffer *b = NULL;

		extract_archive_file_into_buffer(&b, zipfile, zippath,
										 fs_file_get_type(zipfile));

		if (b == NULL) {
			return false;
		}

		if ((ttf = ttf_open_buffer(b->ptr, b->used, size, zippath)) == NULL) {
			buffer_free_weak(b);
			return false;
		}
		buffer_free_weak(b);
	} else {
		ttf_set_pixel_size(ttf, size);
	}

	use_ttf = 1;
	return ttf;
}
#endif

extern BOOL disp_load_zipped_truetype_book_font(const char *ezipfile,
												const char *czipfile,
												const char *ettffile,
												const char *cttffile, int size)
{
#ifdef ENABLE_TTF
	static char prev_ettfpath[PATH_MAX2] = "", prev_ettfarch[PATH_MAX2] = "";
	static char prev_cttfpath[PATH_MAX2] = "", prev_cttfarch[PATH_MAX2] = "";

	use_ttf = 0;
	memset(disp_ewidth, size / 2, 0x80);
	if (book_efont_buffer != NULL && efont_buffer != book_efont_buffer) {
		free((void *) book_efont_buffer);
		book_efont_buffer = NULL;
	}
	if (book_localfont_buffer != NULL && localfont_buffer != book_localfont_buffer) {
		free((void *) book_localfont_buffer);
		book_localfont_buffer = NULL;
	}
	if (ettf != NULL && strcmp(prev_ettfarch, config.ettfarch) == 0
		&& strcmp(prev_ettfpath, config.ettfpath) == 0) {
		ttf_set_pixel_size(ettf, size);
	} else {
		ttf_close(ettf);
		ettf = NULL;
		if (config.ettfarch[0] != '\0') {
			ettf =
				load_archieve_truetype_book_font(config.ettfarch,
												 config.ettfpath,
												 config.bookfontsize);
		} else {
			ettf = ttf_open(config.ettfpath, config.bookfontsize);
		}
		STRCPY_S(prev_ettfarch, config.ettfarch);
		STRCPY_S(prev_ettfpath, config.ettfpath);
	}
	if (cttf != NULL && strcmp(prev_cttfarch, config.cttfarch) == 0
		&& strcmp(prev_cttfpath, config.cttfpath) == 0) {
		ttf_set_pixel_size(cttf, size);
	} else {
		if (cttf != NULL) {
			ttf_close(cttf);
			cttf = NULL;
		}
		if (config.cttfarch[0] != '\0') {
			cttf =
				load_archieve_truetype_book_font(config.cttfarch,
												 config.cttfpath,
												 config.bookfontsize);
		} else {
			cttf = ttf_open(config.cttfpath, config.bookfontsize);
		}
		STRCPY_S(prev_cttfarch, config.cttfarch);
		STRCPY_S(prev_cttfpath, config.cttfpath);
	}

	if (cttf == NULL || ettf == NULL)
		return false;

	load_ttf_config();

	ttf_load_ewidth(ettf, disp_ewidth, 0x80);
	use_ttf = 1;
	return true;
#else
	return false;
#endif
}

#endif


/* END */
